/******************************************************************************
 *  rawstr.cpp   - code for class 'RawStr'- a module that reads raw text
 *				files:  ot and nt using indexs ??.bks ??.cps ??.vss
 *				and provides lookup and parsing functions based on
 *				class StrKey
 */


#include <ctype.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>

#ifndef __GNUC__
#include <io.h>
#else
#include <unistd.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <utilfuns.h>
#include <rawstr.h>


/******************************************************************************
 * RawStr Statics
 */

int RawStr::instance = 0;


/******************************************************************************
 * RawStr Constructor - Initializes data for instance of RawStr
 *
 * ENT:	ipath - path of the directory where data and index files are located.
 *		be sure to include the trailing separator (e.g. '/' or '\')
 *		(e.g. 'modules/texts/rawtext/webster/')
 */

RawStr::RawStr(const char *ipath)
{
	char buf[127];

	nl = '\n';
	lastoff = -1;
	path = 0;
	stdstr(&path, ipath);

#ifndef O_BINARY		// O_BINARY is needed in Borland C++ 4.53
#define O_BINARY 0		// If it hasn't been defined than we probably
#endif				// don't need it.

	sprintf(buf, "%s.idx", path);
	idxfd = open(buf, O_RDONLY|O_BINARY);

	sprintf(buf, "%s.dat", path);
	if ((datfd = open(buf, O_RDONLY|O_BINARY)) < 0) {
		sprintf(buf, "Error: %d", errno);
		perror(buf);
	}

	instance++;
}


/******************************************************************************
 * RawStr Destructor - Cleans up instance of RawStr
 */

RawStr::~RawStr()
{
	if (path)
		delete [] path;

	--instance;

	close(idxfd);
	close(datfd);
}


/******************************************************************************
 * RawStr::getidxbuf	- Gets the index string at the given idx offset
 *						NOTE: buf is allocated and must be freed by
 *							calling function
 *
 * ENT:	ioffset	- offset to lookup
 *		buf		- address of pointer to allocate for storage of string
 */

void RawStr::getidxbuf(long ioffset, char **buf)
{
	int size;
	long offset;
	char ch;
	if ((idxfd > 0) && (datfd > 0)) {
		lseek(idxfd, ioffset, SEEK_SET);
		read(idxfd, &offset, 4);
		lseek(datfd, offset, SEEK_SET);
		for (size = 0; read(datfd, &ch, 1) == 1; size++) {
			if ((ch == '\\') || (ch == 10))
				break;
		}
		*buf = (*buf) ? (char *)realloc(*buf, size + 1) : (char *)malloc(size + 1);
		lseek(datfd, offset, SEEK_SET);
		read(datfd, *buf, size);
		(*buf)[size] = 0;
		for (size--; size > 0; size--)
			(*buf)[size] = toupper((*buf)[size]);
	}
}


/******************************************************************************
 * RawStr::findoffset	- Finds the offset of the key string from the indexes
 *
 * ENT:	key		- key string to lookup
 *		start	- address to store the starting offset
 *		size		- address to store the size of the entry
 */

void RawStr::findoffset(const char *ikey, long *start, short *size)
{
	char *trybuf, *key, quitflag = 0;
	long headoff, tailoff, tryoff;

	if (*ikey) {
		headoff = 6;
		tailoff = lseek(idxfd, 0, SEEK_END) - 6;

		key = new char [ strlen(ikey) + 1 ];
		strcpy(key, ikey);

		for (trybuf = key; *trybuf; trybuf++)
			*trybuf = toupper(*trybuf);

		trybuf = 0;

		while (headoff < tailoff) {
			tryoff = (lastoff == -1) ? headoff + ((((tailoff / 6) - (headoff / 6))) / 2) * 6 : lastoff; 
			lastoff = -1;
			getidxbuf(tryoff, &trybuf);

			if (!strcmp(key, trybuf)) {
				break;
			}

			if (strcmp(key, trybuf) < 0)
				tailoff = (tryoff == headoff) ? headoff : tryoff;
			else headoff = tryoff;
			if (tailoff == headoff + 6) {
				if (quitflag++)
					headoff = tailoff;
			}
		}
		if (headoff >= tailoff)
			tryoff = headoff;
		if (trybuf)
			free(trybuf);
		delete [] key;
	}
	else	tryoff = 0;

	lseek(idxfd, tryoff, SEEK_SET);
	read(idxfd, start, 4);
	read(idxfd, size, 2);
	
	lastoff = tryoff;
}


/******************************************************************************
 * RawStr::preptext	- Prepares the text before returning it to external
 *					objects
 *
 * ENT:	buf	- buffer where text is stored and where to store the prep'd
 *				text.
 */

void RawStr::preptext(char *buf)
{
	char *to, *from, space = 0, cr = 0, realdata = 0, nlcnt = 0;

	for (to = from = buf; *from; from++) {
		switch (*from) {
		case 10:
			if (!realdata)
				continue;
			space = (cr) ? 0 : 1;
			cr = 0;
			nlcnt++;
			if (nlcnt > 1) {
//				*to++ = nl;
				*to++ = nl;
//				nlcnt = 0;
			}
			continue;
		case 13:
			if (!realdata)
				continue;
			*to++ = nl;
			space = 0;
			cr = 1;
			continue;
		}
		realdata = 1;
		nlcnt = 0;
		if (space) {
			space = 0;
			if (*from != ' ') {
				*to++ = ' ';
				from--;
				continue;
			}
		}
		*to++ = *from;
	}
	*to = 0;

	for (to--; to > buf; to--) {			// remove training excess
		if ((*to == 10) || (*to == ' '))
			*to = 0;
		else break;
	}
}


/******************************************************************************
 * RawStr::gettext	- gets text at a given offset
 *
 * ENT:	testmt	- testament file to search in (0 - Old; 1 - New)
 *	start	- starting offset where the text is located in the file
 *	size	- size of text entry
 *	buf	- buffer to store text
 *
 */

void RawStr::gettext(long start, short size, char *idxbuf, char *buf)
{
	char *ch;

	memset(buf, 0, size);
	lseek(datfd, start, SEEK_SET);
	read(datfd, buf, (int)size - 1);

	for (ch = buf; *ch; ch++) {		// skip over index string
		if (*ch == 10) {
			ch++;
			break;
		}
		if (*ch != '\\') {
			*idxbuf = *ch;
			idxbuf++;
		}
	}
	*idxbuf = 0;
	memmove(buf, ch, strlen(ch) + 1);
}
