package org.crosswire.jsword.passage;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Iterator;
import java.util.NoSuchElementException;

import org.crosswire.common.util.Logger;

/**
 * A Passage is a pointer to a single verse. Externally its unique
 * identifier is a String of the form "Gen 1:1" Internally we use
 * <code>int[] { book, chapter, verse }</code>
 *
 * <p>A Verse is designed to be immutable. This is a necessary from a
 * collections point of view. A Verse should always be valid, although
 * some versions may not return any text for verses that they consider to
 * be mis-translated in some way.</p>
 *
 * <p>Optimization information: I spent some time optimizing this class
 * because it is at the heart of things. My benchmark started st 11.25s.
 * By taking the int[] and turning it into 3 ints and it took 10.8s.<br />
 * Cacheing the ordinal number just took the time from 12s to 12s! I guess
 * that the time and extra memory taken up by the extra int overrode the
 * time it saved by repeated queries to the same verse. I guess this would
 * change if we were using a [Ranged|Distinct]Passage instead of a Bitwise
 * Passage (as in the test). Maybe it would be a good idea to have an
 * extra class OrdCacheVerse (or something) that gave us the best of both
 * worlds?<br />
 * Removing the default initialization of the ints took the time down by
 * about 0.25s also.
 * </p>
 * 
 * <p><table border='1' cellPadding='3' cellSpacing='0'>
 * <tr><td bgColor='white' class='TableRowColor'><font size='-7'>
 *
 * Distribution Licence:<br />
 * JSword is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General Public License,
 * version 2 as published by the Free Software Foundation.<br />
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.<br />
 * The License is available on the internet
 * <a href='http://www.gnu.org/copyleft/gpl.html'>here</a>, or by writing to:
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA<br />
 * The copyright to this program is held by it's authors.
 * </font></td></tr></table>
 * @see gnu.gpl.Licence
 * @author Joe Walker [joe at eireneh dot com]
 * @version $Id: Verse.java,v 1.33 2004/09/08 19:55:07 dmsmith Exp $
 */
public final class Verse implements VerseBase, Comparable
{
    /**
     * The default Verse is Genesis 1:1. I didn't want to provide this
     * constructor however, you are supposed to provide a default ctor
     * for all beans. For this reason I suggest you don't use it.
     */
    public Verse()
    {
        originalName = null;

        book = DEFAULT.book;
        chapter = DEFAULT.chapter;
        verse = DEFAULT.verse;
    }

    /**
     * Create a Verse from book, chapter and verse numbers, throwing up
     * if the specified Verse does not exist. This constructor is
     * deliberately package protected so that is used only by VerseFactory.
     * @param original The original verse reference
     * @param book The book number (Genesis = 1)
     * @param chapter The chapter number
     * @param verse The verse number
     * @exception NoSuchVerseException If the reference is illegal
     */
    /*package*/ Verse(String original, int book, int chapter, int verse) throws NoSuchVerseException
    {
        originalName = original;
        set(book, chapter, verse);
    }

    /**
     * Create a Verse from book, chapter and verse numbers, throwing up
     * if the specified Verse does not exist.
     * @param book The book number (Genesis = 1)
     * @param chapter The chapter number
     * @param verse The verse number
     * @exception NoSuchVerseException If the reference is illegal
     */
    public Verse(int book, int chapter, int verse) throws NoSuchVerseException
    {
        this(null, book, chapter, verse);
    }

    /**
     * Create a Verse from book, chapter and verse numbers, patching up if
     * the specified verse does not exist.
     * <p>The actual value of the boolean is ignored. However for future
     * proofing you should only use 'true'. Do not use patch_up=false, use
     * <code>Verse(int, int, int)</code> This so that we can declare this
     * constructor to not throw an exception. Is there a better way of
     * doing this?
     * @param book The book number (Genesis = 1)
     * @param chapter The chapter number
     * @param verse The verse number
     * @param patch_up True to trigger reference fixing
     */
    public Verse(int book, int chapter, int verse, boolean patch_up)
    {
        if (!patch_up)
        {
            throw new IllegalArgumentException(Msg.ERROR_PATCH.toString());
        }

        originalName = null;
        setAndPatch(book, chapter, verse);
    }

    /**
     * Set a Verse using a Verse Ordinal number - WARNING Do not use this
     * method unless you really know the dangers of doing so. Ordinals are
     * not always going to be the same. So you should use a Verse or an
     * int[3] in preference to an int ordinal whenever possible. Ordinal
     * numbers are 1 based and not 0 based.
     * @param ordinal The verse id
     * @exception NoSuchVerseException If the reference is illegal
     */
    public Verse(int ordinal) throws NoSuchVerseException
    {
        originalName = null;
        set(ordinal);
    }

    /**
     * Translate the Passage into a human readable string. This is
     * simply an alias for getName();
     * @return The string representation
     */
    public String toString()
    {
        return getName();
    }

    /**
     * Translate the Passage into a human readable string
     * @return The string representation
     */
    public String getName()
    {
        try
        {
            if (PassageUtil.isPersistentNaming() && originalName != null)
            {
                return originalName;
            }

            // To cope with thing like Jude 2...
            if (BibleInfo.chaptersInBook(book) == 1)
            {
                return BibleInfo.getShortBookName(book)
                    + Verse.VERSE_PREF_DELIM1
                    + verse;
            }
            else
            {
                return BibleInfo.getShortBookName(book)
                    + Verse.VERSE_PREF_DELIM1
                    + chapter
                    + Verse.VERSE_PREF_DELIM2
                    + verse;
            }
        }
        catch (Exception ex)
        {
            assert false : ex;
            return "!Error!"; //$NON-NLS-1$
        }
    }

    /**
     * Translate the Passage into a human readable string
     * @param base The verse to use to cut down unnecessary output.
     * @return The string representation
     */
    public String getName(Verse base)
    {
        if (base == null)
        {
            return getName();
        }

        try
        {
            if (PassageUtil.isPersistentNaming() && originalName != null)
            {
                return originalName;
            }

            // To cope with thing like Jude 2...
            if (BibleInfo.chaptersInBook(book) == 1)
            {
                if (base.book != book)
                {
                    return BibleInfo.getShortBookName(book)
                        + Verse.VERSE_PREF_DELIM1
                        + verse;
                }

                return String.valueOf(verse);
            }
            else
            {
                if (base.book != book)
                {
                    return BibleInfo.getShortBookName(book)
                        + Verse.VERSE_PREF_DELIM1
                        + chapter
                        + Verse.VERSE_PREF_DELIM2
                        + verse;
                }

                if (base.chapter != chapter)
                {
                    return chapter
                        + Verse.VERSE_PREF_DELIM2
                        + verse;
                }

                return String.valueOf(verse);
            }
        }
        catch (Exception ex)
        {
            assert false : ex;
            return "!Error!"; //$NON-NLS-1$
        }
    }

    /**
     * The OSIS defined specification for this Verse.
     * Uses short books names, with "." as a verse part separator.
     * @return a String containing the OSIS description of the verses
     */
    public String getOSISName()
    {
        try
        {
            return BibleInfo.getOSISName(book)
                + Verse.VERSE_OSIS_DELIM
                + chapter
                + Verse.VERSE_OSIS_DELIM
                + verse;
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return "!Error!"; //$NON-NLS-1$
        }
    }

    /**
     * Get a copy of ourselves. Points to note:
     *   Call clone() not new() on member Objects, and on us.
     *   Do not use Copy Constructors! - they do not inherit well.
     *   Think about this needing to be synchronized
     *   If this is not cloneable then writing cloneable children is harder
     * @return A complete copy of ourselves
     * @exception CloneNotSupportedException We don't do this but our kids might
     */
    public Object clone() throws CloneNotSupportedException
    {
        Verse copy = (Verse) super.clone();

        copy.book = book;
        copy.chapter = chapter;
        copy.verse = verse;
        //copy.ord = ord;
        copy.originalName = originalName;

        return copy;
    }

    /**
     * Is this Object equal to us. Points to note:
     *   If you override equals(), you must override hashCode() too.
     *   If you are doing this it is a good idea to be immutable.
     * @param obj The thing to test against
     * @return True/False is we are or are not equal to obj
     */
    public boolean equals(Object obj)
    {
        // Since this can not be null
        if (obj == null)
        {
            return false;
        }

        // Check that that is the same as this
        // Don't use instanceOf since that breaks inheritance
        if (!obj.getClass().equals(this.getClass()))
        {
            return false;
        }

        Verse v = (Verse) obj;

        // The real tests
        if (v.getBook() != getBook())
        {
            return false;
        }

        if (v.getChapter() != getChapter())
        {
            return false;
        }

        if (v.getVerse() != getVerse())
        {
            return false;
        }

        return true;
    }

    /**
     * This returns the ordinal number of the verse
     * so <code>new Verse("Rev 22:21").hashCode() = 31104</code>.
     * <p><b>However should should not reply on this being true</b>
     * @return The hashing number
     */
    public int hashCode()
    {
        return getOrdinal();
    }

    /**
     * Compare this to a given object
     * @param obj The thing to compare against
     * @return 1 means he is earlier than me, -1 means he is later ...
     */
    public int compareTo(Object obj)
    {
        Verse that = null;
        if (obj instanceof Verse)
        {
            that = (Verse) obj;
        }
        else
        {
            that = ((VerseRange) obj).getStart();
        }

        int thatStart = that.getOrdinal();
        int thisStart = this.getOrdinal();

        if (thatStart > thisStart)
        {
            return -1;
        }

        if (thatStart < thisStart)
        {
            return 1;
        }

        return 0;
    }

    /**
     * Is this verse adjacent to another verse
     * @param that The thing to compare against
     * @return 1 means he is earlier than me, -1 means he is later ...
     */
    public boolean adjacentTo(Verse that)
    {
        return Math.abs(that.getOrdinal() - getOrdinal()) == 1;
    }

    /**
     * How many verses are there in between the 2 Verses.
     * The answer is -ve if that is bigger than this.
     * The answer is inclusive of that and exclusive of this, so that
     * <code>gen11.difference(gen12) == 1</code>
     * @param that The Verse to compare this to
     * @return The count of verses between this and that.
     */
    public int subtract(Verse that)
    {
        return getOrdinal() - that.getOrdinal();
    }

    /**
     * Get the verse n down from here this Verse.
     * @param n The number to count down by
     * @return The new Verse
     */
    public Verse subtract(int n)
    {
        try
        {
            int new_ordinal = Math.max(getOrdinal() - n, 1);
            return new Verse(new_ordinal);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return Verse.DEFAULT;
        }
    }

    /**
     * Get the verse that is a few verses on from the one
     * we've got.
     * @param extra the number of verses later than the one we're one
     * @return The new verse
     */
    public Verse add(int extra)
    {
        try
        {
            int new_ordinal = Math.min(getOrdinal() + extra, BibleInfo.versesInBible());
            return new Verse(new_ordinal);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return Verse.DEFAULT;
        }
    }

    /**
     * Return the book that we refer to
     * @return The book number (Genesis = 1)
     */
    public int getBook()
    {
        return book;
    }

    /**
     * Return the chapter that we refer to
     * @return The chapter number
     */
    public int getChapter()
    {
        return chapter;
    }

    /**
     * Return the verse that we refer to
     * @return The verse number
     */
    public int getVerse()
    {
        return verse;
    }

    /**
     * Is this verse the first in a chapter
     * @return true or false ...
     */
    public boolean isStartOfChapter()
    {
        return verse == 1;
    }

    /**
     * Is this verse the first in a chapter
     * @return true or false ...
     */
    public boolean isEndOfChapter()
    {
        try
        {
            return verse == BibleInfo.versesInChapter(book, chapter);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return true;
        }
    }

    /**
     * Is this verse the first in a chapter
     * @return true or false ...
     */
    public boolean isStartOfBook()
    {
        return verse == 1 && chapter == 1;
    }

    /**
     * Is this verse the first in a chapter
     * @return true or false ...
     */
    public boolean isEndOfBook()
    {
        try
        {
            return verse == BibleInfo.versesInChapter(book, chapter)
                && chapter == BibleInfo.chaptersInBook(book);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return true;
        }
    }

    /**
     * Is this verse in the same chapter as that one
     * @param that The verse to compate to
     * @return true or false ...
     */
    public boolean isSameChapter(Verse that)
    {
        return book == that.book && chapter == that.chapter;
    }

    /**
     * Is this verse in the same book as that one
     * @param that The verse to compate to
     * @return true or false ...
     */
    public boolean isSameBook(Verse that)
    {
        return book == that.book;
    }

    /**
     * Return the verse that we refer to
     * @return An array of 3 ints 0=book, 1=chapter, 2=verse
     */
    public int[] getRefArray()
    {
        return new int[] { book, chapter, verse };
    }

    /**
     * Return the verse id that we refer to, where Gen 1:1 = 1, and
     * Rev 22:21 = 31104
     * @return The verse number
     */
    public int getOrdinal()
    {
        try
        {
            return BibleInfo.verseOrdinal(book, chapter, verse);
        }
        catch (NoSuchVerseException ex)
        {
            // A verse should never be illegal so
            log.error("ref=" + book + ", " + chapter + ", " + verse); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            assert false : ex;
            return 1;
        }
    }

    /**
     * Return the bigger of the 2 verses. If the verses are equal()
     * then return Verse a
     * @param a The first verse to compare
     * @param b The second verse to compare
     * @return The bigger of the 2 verses
     */
    public static final Verse max(Verse a, Verse b)
    {
        if (a.compareTo(b) == -1)
        {
            return b;
        }
        else
        {
            return a;
        }
    }

    /**
     * Return the smaller of the 2 verses. If the verses are equal()
     * then return Verse a
     * @param a The first verse to compare
     * @param b The second verse to compare
     * @return The smaller of the 2 verses
     */
    public static final Verse min(Verse a, Verse b)
    {
        if (a.compareTo(b) == 1)
        {
            return b;
        }
        else
        {
            return a;
        }
    }

    /**
     * Create an array of Verses
     * @return The array of verses that this makes up
     */
    public Verse[] toVerseArray()
    {
        return new Verse[] { this };
    }

    /**
     * Enumerate over the verse in this verse!.
     * This may seem silly, however is is very useful to be able to treat
     * Verses and Ranges the same (VerseBase) and this is a common accessor.
     * @return A verse iterator
     * @see org.crosswire.jsword.passage.VerseBase#verseIterator()
     */
    public Iterator verseIterator()
    {
        return new VerseIterator();
    }

    /**
     * Create a new Verse being the last verse in the current book
     * @return The last verse in this book
     */
    public Verse getLastVerseInBook()
    {
        try
        {
            int lastchap = BibleInfo.chaptersInBook(book);
            int lastverse = BibleInfo.versesInChapter(book, lastchap);

            return new Verse(book, lastchap, lastverse);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return Verse.DEFAULT;
        }
    }

    /**
     * Create a new Verse being the last verse in the current book
     * @return The last verse in this book
     */
    public Verse getLastVerseInChapter()
    {
        try
        {
            int lastverse = BibleInfo.versesInChapter(book, chapter);

            return new Verse(book, chapter, lastverse);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return Verse.DEFAULT;
        }
    }

    /**
     * Create a new Verse being the first verse in the current book
     * @return The first verse in this book
     */
    public Verse getFirstVerseInBook()
    {
        try
        {
            return new Verse(book, 1, 1);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return Verse.DEFAULT;
        }
    }

    /**
     * Create a new Verse being the first verse in the current book
     * @return The first verse in this book
     */
    public Verse getFirstVerseInChapter()
    {
        try
        {
            return new Verse(book, chapter, 1);
        }
        catch (NoSuchVerseException ex)
        {
            assert false : ex;
            return Verse.DEFAULT;
        }
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#getParent()
     */
    public Key getParent()
    {
        return parent;
    }

    /**
     * Set a parent Key. This allows us to follow the Key interface more
     * closely, although the concept of a parent for a verse is fairly
     * alien.
     * @param parent The parent Key for this verse
     */
    public void setParent(Key parent)
    {
        this.parent = parent;
    }

    /**
     * This is simply a convenience function to wrap Integer.parseInt()
     * and give us a reasonable exception on failure. It is called by
     * VerseRange hence protected, however I would prefer private
     * @param text The string to be parsed
     * @return The correctly parsed chapter or verse
     * @exception NoSuchVerseException If the reference is illegal
     */
    protected static int parseInt(String text) throws NoSuchVerseException
    {
        try
        {
            return Integer.parseInt(text);
        }
        catch (NumberFormatException ex)
        {
            throw new NoSuchVerseException(Msg.VERSE_PARSE, new Object[] { text });
        }
    }

    /**
     * Mutate into this reference and fix the reference if needed.
     * This nust only be called from a ctor to maintain immutability
     * @param book The book to set (Genesis = 1)
     * @param chapter The chapter to set
     * @param verse The verse to set
     */
    private final void setAndPatch(int book, int chapter, int verse)
    {
        int[] ref = { book, chapter, verse };

        BibleInfo.patch(ref);

        this.book = ref[BOOK];
        this.chapter = ref[CHAPTER];
        this.verse = ref[VERSE];
    }

    /*
     * Mutate into this reference and fix the reference if needed.
     * This must only be called from a ctor to maintain immutability
     * @param ref An array of the book, chapter and verse to set
     *
    private final void setAndPatch(int[] ref)
    {
        setAndPatch(ref[BOOK], ref[CHAPTER], ref[VERSE]);
    }
    */

    /**
     * Verify and set the references.
     * This must only be called from a ctor to maintain immutability
     * @param book The book to set (Genesis = 1)
     * @param chapter The chapter to set
     * @param verse The verse to set
     * @exception NoSuchVerseException If the verse can not be understood
     */
    private final void set(int book, int chapter, int verse) throws NoSuchVerseException
    {
        BibleInfo.validate(book, chapter, verse);

        this.book = book;
        this.chapter = chapter;
        this.verse = verse;
    }

    /**
     * Set the references.
     * This must only be called from a ctor to maintain immutability
     * @param ordinal The ordinal of the verse
     * @exception NoSuchVerseException If the verse can not be understood
     */
    private final void set(int ordinal) throws NoSuchVerseException
    {
        int[] ref = BibleInfo.decodeOrdinal(ordinal);

        book = ref[BOOK];
        chapter = ref[CHAPTER];
        verse = ref[VERSE];
    }

    /**
     * Write out the object to the given ObjectOutputStream
     * @param out The stream to write our state to
     * @throws IOException if the read fails
     * @serialData Write the ordinal number of this verse
     */
    private void writeObject(ObjectOutputStream out) throws IOException
    {
        // Call even if there is no default serializable fields.
        out.defaultWriteObject();

        // save the ordinal of the verse
        out.writeInt(getOrdinal());

        // Ignore the original name. Is this wise?
        // I am expecting that people are not that fussed about it and
        // it could make everything far more verbose
    }

    /**
     * Write out the object to the given ObjectOutputStream
     * @param in The stream to read our state from
     * @throws IOException if the read fails
     * @throws ClassNotFoundException If the read data is incorrect
     * @serialData Write the ordinal number of this verse
     */
    private void readObject(ObjectInputStream in) throws IOException, ClassNotFoundException
    {
        // Call even if there is no default serializable fields.
        in.defaultReadObject();

        try
        {
            set(in.readInt());
        }
        catch (NoSuchVerseException ex)
        {
            throw new IOException(ex.getMessage());
        }

        // We are ignoring the originalName. It was set to null in the
        // default ctor so I will ignore it here.
    }

    /**
     * Iterator over 1 verse - For being a VerseBase.
     */
    private class VerseIterator implements Iterator
    {
        /* (non-Javadoc)
         * @see java.util.Iterator#hasNext()
         */
        public boolean hasNext()
        {
            return !done;
        }

        /* (non-Javadoc)
         * @see java.util.Iterator#next()
         */
        public Object next()
        {
            if (done)
            {
                throw new NoSuchElementException();
            }

            done = true;
            return Verse.this;
        }

        /* (non-Javadoc)
         * @see java.util.Iterator#remove()
         */
        public void remove() throws UnsupportedOperationException
        {
            throw new UnsupportedOperationException();
        }

        private boolean done;
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#canHaveChildren()
     */
    public boolean canHaveChildren()
    {
        return false;
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#size()
     */
    public int getChildCount()
    {
        return 0;
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#isEmpty()
     */
    public boolean isEmpty()
    {
        return true;
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#contains(org.crosswire.jsword.passage.Key)
     */
    public boolean contains(Key key)
    {
        return false;
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#iterator()
     */
    public Iterator iterator()
    {
        return new Iterator()
        {
            public void remove()
            {
                throw new UnsupportedOperationException();
            }

            public boolean hasNext()
            {
                return false;
            }

            public Object next()
            {
                return null;
            }
        };
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#add(org.crosswire.jsword.passage.Key)
     */
    public void addAll(Key key)
    {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#remove(org.crosswire.jsword.passage.Key)
     */
    public void removeAll(Key key)
    {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#retain(org.crosswire.jsword.passage.Key)
     */
    public void retainAll(Key key)
    {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#clear()
     */
    public void clear()
    {
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#get(int)
     */
    public Key get(int index)
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#indexOf(org.crosswire.jsword.passage.Key)
     */
    public int indexOf(Key that)
    {
        return -1;
    }

    /* (non-Javadoc)
     * @see org.crosswire.jsword.passage.Key#blur(int)
     */
    public void blur(int by, RestrictionType restrict)
    {
        throw new UnsupportedOperationException();
    }

    /**
     * To make serialization work across new versions
     */
    static final long serialVersionUID = -4033921076023185171L;

    /**
     * To make the code more readible, the book is the first part of a int[]
     */
    private static final int BOOK = 0;

    /**
     * To make the code more readible, the chapter is the second part of a int[]
     */
    private static final int CHAPTER = 1;

    /**
     * To make the code more readible, the verse is the third part of a int[]
     */
    private static final int VERSE = 2;

    /**
     * What characters should we use to separate parts of an OSIS verse reference 
     */
    public static final String VERSE_OSIS_DELIM = "."; //$NON-NLS-1$

    /**
     * What characters should we use to separate the book from the chapter
     */
    public static final String VERSE_PREF_DELIM1 = " "; //$NON-NLS-1$

    /**
     * What characters should we use to separate the chapter from the verse
     */
    public static final String VERSE_PREF_DELIM2 = ":"; //$NON-NLS-1$

    /**
     * The default verse
     */
    public static final Verse DEFAULT = new Verse(1, 1, 1, true);

    /**
     * The parent key. See the key interface for more information.
     * NOTE(joe): These keys are not serialized, should we?
     * @see Key
     */
    private transient Key parent;

    /**
     * The book number. Genesis=1
     */
    private transient int book;

    /**
     * The chapter number
     */
    private transient int chapter;

    /**
     * The verse number
     */
    private transient int verse;

    /**
     * The original string for picky users
     */
    private transient String originalName;

    /**
     * The log stream
     */
    private static final Logger log = Logger.getLogger(Verse.class);
}
